from django.db import migrations

def populate_home(apps, schema_editor):
    Page = apps.get_model('pages', 'Page')
    Section = apps.get_model('sections', 'Section')

    # 1. Récupération de la Home
    try:
        home_page = Page.objects.get(slug='accueil')
    except Page.DoesNotExist:
        return # Sécurité

    # 2. Nettoyage (On repart de zéro pour cette page uniquement)
    Section.objects.filter(page=home_page).delete()

    # =========================================================
    # SECTION 1 : HERO SLIDER (L'Immersion)
    # =========================================================
    Section.objects.create(
        page=home_page,
        type='hero_slider',
        internal_name="Hero Principal",
        order=1,
        data={
            "autoplay": True,
            "interval": 6000,
            "slides": [
                {
                    "title": "Sculpter le Futur Digital.",
                    "subtitle": "Nous transformons vos défis complexes en expériences web performantes et inoubliables.",
                    "tag": "Innovation",
                    "btn_primary_text": "Voir nos réalisations",
                    "btn_primary_url": "/portfolio",
                    "btn_secondary_text": "L'Agence",
                    "btn_secondary_url": "/a-propos"
                },
                {
                    "title": "Expertise Technique Sans Faille.",
                    "subtitle": "De l'architecture Cloud au développement Headless, nous maîtrisons la stack moderne.",
                    "tag": "Tech",
                    "btn_primary_text": "Nos Services",
                    "btn_primary_url": "/services"
                },
                {
                    "title": "Votre Croissance, Notre Priorité.",
                    "subtitle": "Des stratégies digitales conçues pour maximiser votre impact et votre ROI.",
                    "tag": "Performance",
                    "btn_primary_text": "Démarrer un projet",
                    "btn_primary_url": "/contact"
                }
            ]
        }
    )

    # =========================================================
    # SECTION 2 : LOGO CLOUD (Preuve Sociale - Placeholders)
    # =========================================================
    Section.objects.create(
        page=home_page,
        type='logo_cloud',
        internal_name="Logos Partenaires",
        order=2,
        data={
            "logos": [
                {"name": "TechCorp", "image": "/media/defaults/logo1.png"},
                {"name": "Innovate", "image": "/media/defaults/logo2.png"},
                {"name": "FutureLabs", "image": "/media/defaults/logo3.png"},
                {"name": "GlobalDesign", "image": "/media/defaults/logo4.png"},
                {"name": "StartUp Flow", "image": "/media/defaults/logo5.png"}
            ]
        }
    )

    # =========================================================
    # SECTION 3 : MANIFESTE (Vision)
    # =========================================================
    Section.objects.create(
        page=home_page,
        type='text_block',
        internal_name="Manifeste Agence",
        order=3,
        data={
            "align": "center",
            "content": """
            <h2>Plus qu'une agence, un partenaire technologique.</h2>
            <p>Dans un monde numérique saturé, la médiocrité n'est pas une option. 
            Nous combinons <strong>design émotionnel</strong> et <strong>rigueur technique</strong> 
            pour créer des produits qui ne se contentent pas de fonctionner, mais qui inspirent.</p>
            """
        }
    )

    # =========================================================
    # SECTION 4 : BENTO GRID (Forces)
    # =========================================================
    Section.objects.create(
        page=home_page,
        type='features',
        internal_name="Points Forts (Bento)",
        order=4,
        data={
            "title": "Pourquoi nous choisir ?",
            "items": [
                {
                    "title": "Architecture Headless",
                    "description": "Séparation du backend et du frontend pour une sécurité et une flexibilité totales.",
                    "col_span": 2, "row_span": 1,
                    "icon": "fa-server"
                },
                {
                    "title": "Design System",
                    "description": "Une cohérence visuelle parfaite sur tous les écrans.",
                    "col_span": 1, "row_span": 1,
                    "icon": "fa-paint-brush"
                },
                {
                    "title": "Performance Green",
                    "description": "Code optimisé pour réduire l'empreinte carbone et charger en <0.5s.",
                    "col_span": 1, "row_span": 1,
                    "icon": "fa-leaf"
                },
                {
                    "title": "Sécurité Avancée",
                    "description": "Protection des données et conformité RGPD native.",
                    "col_span": 2, "row_span": 1,
                    "icon": "fa-lock"
                }
            ]
        }
    )

    # =========================================================
    # SECTION 5 : STATS (Tes chiffres)
    # =========================================================
    Section.objects.create(
        page=home_page,
        type='stats',
        internal_name="Chiffres Clés",
        order=5,
        data={
            "style": "dark",
            "items": [
                {"value": "10", "label": "Années d'Expérience", "suffix": "+"},
                {"value": "20", "label": "Projets d'Envergure", "suffix": "+"},
                {"value": "100", "label": "Engagement ROI", "suffix": "%"}
            ]
        }
    )

    # =========================================================
    # SECTION 6 : ACCORDION (FAQ/Expertise)
    # =========================================================
    Section.objects.create(
        page=home_page,
        type='accordion',
        internal_name="FAQ Accueil",
        order=6,
        data={
            "title": "Questions Fréquentes",
            "items": [
                {
                    "question": "Combien de temps pour développer un projet ?",
                    "answer": "Cela dépend de la complexité. Un site vitrine prend entre 4 à 6 semaines, tandis qu'une application métier complexe peut demander 3 à 6 mois."
                },
                {
                    "question": "Travaillez-vous avec des startups ?",
                    "answer": "Absolument. Nous adorons l'agilité des startups et nous adaptons nos process pour livrer des MVP robustes rapidement."
                },
                {
                    "question": "Assurez-vous la maintenance ?",
                    "answer": "Oui. Nous ne vous laissons pas seul après le lancement. Nous proposons des contrats de maintenance évolutive et corrective."
                }
            ]
        }
    )

    # =========================================================
    # SECTION 7 : TÉMOIGNAGES (Placeholders)
    # =========================================================
    Section.objects.create(
        page=home_page,
        type='testimonials',
        internal_name="Avis Clients",
        order=7,
        data={
            "items": [
                {
                    "author_name": "Marc L.",
                    "author_role": "CEO, FinTech Solutions",
                    "quote": "Une équipe proactive qui a su comprendre nos enjeux métier au-delà du simple code.",
                    "rating": 5
                },
                {
                    "author_name": "Julie D.",
                    "author_role": "Dir. Marketing, Retail Group",
                    "quote": "Le redesign de notre plateforme a augmenté notre taux de conversion de 40% en 3 mois.",
                    "rating": 5
                }
            ]
        }
    )

    # =========================================================
    # SECTION 8 : CTA FINAL (Conversion)
    # =========================================================
    Section.objects.create(
        page=home_page,
        type='cta',
        internal_name="CTA Bas de Page",
        order=8,
        data={
            "title": "Prêt à passer au niveau supérieur ?",
            "text": "Ne laissez pas votre projet digital au hasard. Discutons de vos objectifs.",
            "btn_text": "Lancer le projet",
            "btn_url": "/contact"
        }
    )

def reverse_populate_home(apps, schema_editor):
    pass

class Migration(migrations.Migration):

    dependencies = [
        ('pages', '0003_create_detail_pages'), # S'assure que la Home existe déjà
        ('sections', '0001_initial'),
    ]

    operations = [
        migrations.RunPython(populate_home, reverse_populate_home),
    ]