from django.db import migrations
from django.utils.text import slugify

def populate_content(apps, schema_editor):
    # Modèles
    Article = apps.get_model('content', 'Article')
    Category = apps.get_model('content', 'Category')
    Page = apps.get_model('pages', 'Page')
    Section = apps.get_model('sections', 'Section')
    User = apps.get_model('users', 'User')

    # 1. Création de l'utilisateur auteur par défaut (si inexistant)
    author, _ = User.objects.get_or_create(username='admin', defaults={'email': 'admin@example.com'})

    # 2. Création de la catégorie Portfolio
    cat_portfolio, _ = Category.objects.get_or_create(name="Portfolio", slug="portfolio")

    # =========================================================
    # PARTIE 1 : LES PROJETS (Articles)
    # =========================================================
    projects_data = [
        {
            "title": "IMAGIN_K PLATFORM // ERP SaaS",
            "excerpt": "Système nerveux central pour atelier digital. Pipeline Kanban, Facturation PDF et Comptabilité intégrée.",
            "content": """
            <p>Conception et développement de l'architecture centrale du SaaS "IMAGIN_K". Ce système ERP/CRM complet pilote l'intégralité de l'activité de l'atelier digital.</p>
            <h3>MODULES DÉPLOYÉS :</h3>
            <ul>
                <li><strong>Automatisation du cycle client :</strong> Pipeline visuel Kanban pour le suivi de production.</li>
                <li><strong>Système Financier :</strong> Génération dynamique de factures PDF et module de comptabilité analytique.</li>
                <li><strong>Passerelle de Paiement :</strong> Intégration sécurisée de l'API FedaPay avec gestion des Webhooks.</li>
                <li><strong>Portail Partenaires :</strong> Interface dédiée pour la gestion des collaborations externes.</li>
            </ul>
            <p><strong>Stack :</strong> Django, Python, PostgreSQL, API REST, Bootstrap 5.</p>
            """,
            "status": "published",
            "published_at": "2025-08-13",
        },
        {
            "title": "SECURISATION PROTOCOL // EQUILIBRE",
            "excerpt": "Migration critique Legacy vers Django. Patch de failles de sécurité et Dashboard Admin sur mesure.",
            "content": """
            <p>Opération de refonte majeure pour "Equilibre Media". L'objectif était de migrer une infrastructure WordPress vieillissante vers une architecture Django robuste.</p>
            <h3>INTERVENTIONS TECHNIQUES :</h3>
            <ul>
                <li>Migration complète des données et de la logique métier.</li>
                <li>Résolution d'une faille critique (Injection/Spam) sur les formulaires de contact.</li>
                <li>Développement d'un Dashboard d'administration custom.</li>
            </ul>
            <p><strong>Stack :</strong> Django, Python, Git/GitHub.</p>
            """,
            "status": "published",
            "published_at": "2024-06-01",
        },
        {
            "title": "BELLE ALLURE // E-COM",
            "excerpt": "Plateforme e-commerce sur mesure pour le prêt-à-porter. UX fluide et gestion catalogue.",
            "content": """
            <p>Développement d'une solution e-commerce "Boutique de Prêt-à-porter" sans CMS générique. Architecture optimisée pour la performance.</p>
            <h3>FONCTIONNALITÉS :</h3>
            <ul>
                <li>Gestion avancée du catalogue produits (Variantes, Tailles, Couleurs).</li>
                <li>Tunnel de commande optimisé.</li>
                <li>Interface responsive mobile-first.</li>
            </ul>
            <p><strong>Stack :</strong> Django, HTML5/CSS3, JavaScript.</p>
            """,
            "status": "published",
            "published_at": "2025-09-01",
        },
        {
            "title": "LE COLISEE // LUXURY E-COM",
            "excerpt": "Plateforme e-commerce de niche avec Concierge IA (Mistral) et algorithme de recommandation.",
            "content": """
            <p>Conception et développement d'une architecture e-commerce sur-mesure pour l'univers de la parfumerie de niche.</p>
            <p><strong>Architecture Backend Haute Performance :</strong></p>
            <ul>
                <li>Structure monolithique modulaire composée de <strong>10 applications Django distinctes</strong>.</li>
                <li>Modélisation de données complexe (+25 modèles).</li>
            </ul>
            <p><strong>Fonctionnalités "Next-Gen" :</strong></p>
            <ul>
                <li><strong>IA Concierge "Benny" :</strong> Agent RAG connecté à Mistral AI.</li>
                <li><strong>Algorithme de Recommandation :</strong> QCM pondéré pour suggérer le parfum idéal.</li>
            </ul>
            <p><strong>Stack :</strong> Python, Django, PostgreSQL, API REST, IA & LLMs.</p>
            """,
            "status": "published",
            "published_at": "2025-12-01",
        },
         {
            "title": "DYNAMIC PORTFOLIO // ENGINE",
            "excerpt": "Moteur de portfolio générique et réutilisable. Architecture MVT stricte.",
            "content": """
            <p>Développement d'un moteur de site web générique ("Portfolio Engine") conçu pour être déployé rapidement.</p>
            <p><strong>Architecture du Système :</strong></p>
            <ul>
                <li><strong>100% Dynamique :</strong> Tout provient de la base de données.</li>
                <li><strong>Séparation des responsabilités :</strong> Découpage strict par applications.</li>
                <li><strong>Interface d'Administration :</strong> Configuration UX de l'admin Django.</li>
            </ul>
            <p><strong>Stack :</strong> Django, Python, HTML5/CSS3, PostgreSQL.</p>
            """,
            "status": "published",
            "published_at": "2025-05-15",
        }
    ]

    for proj in projects_data:
        Article.objects.get_or_create(
            slug=slugify(proj['title']),
            defaults={
                **proj,
                'category': cat_portfolio,
                'author': author
            }
        )

    # =========================================================
    # PARTIE 2 : LES SERVICES (Pages + Sections)
    # =========================================================
    
    services_list = [
        {
            "title": "Architecture Backend & API",
            "slug": "service-backend-api",
            "desc_short": "Conception de systèmes robustes, sécurisés et d'API RESTful haute performance.",
            "desc_html": """
            <p>Le backend est le moteur invisible qui propulse votre business. Je conçois des architectures logicielles <strong>scalables et résilientes</strong>, capables de supporter la croissance de votre activité sans dette technique.</p>
            <p>Mon approche privilégie la clarté du code, la sécurité des données et la performance brute, en utilisant la puissance de Python et Django.</p>
            """,
            "features": [
                {"title": "API RESTful (DRF)", "description": "Architecture standardisée", "icon": "fa-server"},
                {"title": "Microservices", "description": "Ou Monolithe Modulaire", "icon": "fa-cubes"},
                {"title": "Sécurité (JWT/OAuth2)", "description": "Protection des données", "icon": "fa-lock"},
                {"title": "PostgreSQL Tuning", "description": "Optimisation DB", "icon": "fa-database"}
            ]
        },
        {
            "title": "Développement SaaS & Web",
            "slug": "service-saas-web",
            "desc_short": "Transformation de vos idées en plateformes digitales complètes.",
            "desc_html": """
            <p>Je transforme des concepts complexes en <strong>applications web intuitives</strong>. Spécialisé dans le développement SaaS (Software as a Service), je livre des solutions "clés en main" prêtes pour le marché.</p>
            <p>Que ce soit pour un MVP rapide ou une plateforme d'entreprise massive, je garantis une expérience utilisateur fluide et une administration simplifiée.</p>
            """,
            "features": [
                {"title": "Full-Stack Django", "description": "Solutions complètes", "icon": "fa-laptop-code"},
                {"title": "Dashboards Custom", "description": "Outils de pilotage", "icon": "fa-chart-line"},
                {"title": "Paiements (Stripe/FedaPay)", "description": "Monétisation intégrée", "icon": "fa-credit-card"},
                {"title": "Déploiement Cloud", "description": "Docker & CI/CD", "icon": "fa-cloud-upload-alt"}
            ]
        },
        {
            "title": "Solutions IA & Automatisation",
            "slug": "service-ia-rag",
            "desc_short": "Intégration d'agents conversationnels et automatisation de processus métier.",
            "desc_html": """
            <p>Faites entrer votre entreprise dans l'ère cognitive. J'intègre des <strong>modèles de langage (LLM)</strong> directement dans vos processus métiers pour automatiser le support client, l'analyse de données ou la génération de contenu.</p>
            <p>Expertise spécifique en <strong>RAG (Retrieval-Augmented Generation)</strong> pour connecter l'IA à vos propres données d'entreprise en toute sécurité.</p>
            """,
            "features": [
                {"title": "Chatbots Contextuels", "description": "Agents intelligents", "icon": "fa-robot"},
                {"title": "Intégration LLM", "description": "OpenAI, Mistral, Llama", "icon": "fa-brain"},
                {"title": "Automatisation Python", "description": "Scripts métiers", "icon": "fa-cogs"},
                {"title": "Scraping de données", "description": "Extraction intelligente", "icon": "fa-spider"}
            ]
        },
        {
            "title": "E-Commerce & ERP Sur Mesure",
            "slug": "service-ecommerce-erp",
            "desc_short": "Solutions de vente en ligne et outils de gestion interne adaptés à 100%.",
            "desc_html": """
            <p>Oubliez les limites des CMS standards. Je développe des solutions de commerce et de gestion <strong>entièrement personnalisées</strong> qui s'adaptent à vos processus, et non l'inverse.</p>
            <p>De la gestion de stock complexe à la facturation automatisée, chaque module est pensé pour optimiser votre productivité et vos ventes.</p>
            """,
            "features": [
                {"title": "Tunnels de vente", "description": "Conversion optimisée", "icon": "fa-funnel-dollar"},
                {"title": "Facturation Auto", "description": "Génération PDF", "icon": "fa-file-invoice"},
                {"title": "CRM Custom", "description": "Gestion relation client", "icon": "fa-users"},
                {"title": "Gestion de Stocks", "description": "Inventaire temps réel", "icon": "fa-boxes"}
            ]
        }
    ]

    for svc in services_list:
        # A. Créer la Page
        page, _ = Page.objects.get_or_create(
            slug=svc['slug'],
            defaults={
                "title": svc['title'],
                "meta_description": svc['desc_short'],
                "is_published": True
            }
        )
        
        # Nettoyage préventif
        Section.objects.filter(page=page).delete()

        # B. Section 1 : Hero Simple
        Section.objects.create(
            page=page, type='hero_simple', order=1, internal_name="Hero",
            data={"title": svc['title'], "subtitle": svc['desc_short']}
        )

        # C. Section 2 : Description HTML (Text Block)
        Section.objects.create(
            page=page, type='text_block', order=2, internal_name="Description Détail",
            data={"content": svc['desc_html'], "align": "left"}
        )

        # D. Section 3 : Features (Bento Grid adapté)
        # On adapte le format Features pour qu'il colle au serializer
        features_items = []
        for feat in svc['features']:
            features_items.append({
                "title": feat['title'],
                "description": feat['description'],
                "icon": feat['icon'],
                "col_span": 1, "row_span": 1
            })
            
        Section.objects.create(
            page=page, type='features', order=3, internal_name="Caractéristiques",
            data={"title": "Ce que ce service inclut", "items": features_items}
        )
        
        # E. Section 4 : CTA
        Section.objects.create(
            page=page, type='cta', order=4, internal_name="CTA",
            data={
                "title": "Besoin de cette expertise ?",
                "text": "Parlons de votre projet.",
                "btn_text": "Me contacter",
                "btn_url": "/contact"
            }
        )

def reverse_populate_content(apps, schema_editor):
    pass

class Migration(migrations.Migration):

    dependencies = [
        ('content', '0001_initial'),
        ('pages', '0004_populate_home_content'), # Dépend de la structure page
        ('users', '0001_initial'),
    ]

    operations = [
        migrations.RunPython(populate_content, reverse_populate_content),
    ]