import logging
from django.conf import settings
from openai import OpenAI
from django.db.models import Q

# Imports des modèles de nos applications
from apps.pages.models import Page
from apps.content.models import Article
from apps.settings_site.models import GlobalSettings
from apps.chatbot.models import Conversation, ChatMessage
from . import prompts

logger = logging.getLogger(__name__)

# Initialisation du client OpenRouter
client = OpenAI(
    base_url="https://openrouter.ai/api/v1",
    api_key=settings.OPENROUTER_API_KEY,
    default_headers={
        "HTTP-Referer": "http://localhost:8000", # À adapter en prod
        "X-Title": "CMS Headless Chatbot",
    },
)

def get_global_context():
    """Récupère les infos globales (Email, Tel, Nom du site)"""
    config = GlobalSettings.objects.first()
    if not config:
        return "Nom du site: Mon Agence."
    
    return f"""
    INFOS GLOBALES DU SITE:
    Nom: {config.site_name}
    Email contact: {config.contact_email}
    Téléphone: {config.contact_phone}
    Adresse: {config.contact_address}
    """

def find_context_in_db(user_query: str) -> str:
    """
    RAG Simplifié : Recherche par mots-clés dans les Pages et Articles.
    Pour une version avancée, on utiliserait pgvector (Embeddings).
    """
    query = user_query.lower()
    context_parts = []
    
    # 1. Toujours injecter les infos globales (Contact, Nom)
    context_parts.append(get_global_context())

    # 2. Recherche dans les Pages (Titres et Descriptions)
    # On cherche les pages dont le titre ou la desc contient un mot de la requête
    relevant_pages = Page.objects.filter(
        is_published=True
    ).filter(
        Q(title__icontains=query) | 
        Q(meta_description__icontains=query) |
        Q(slug__icontains=query)
    )[:3] # On limite à 3 résultats pour ne pas saturer le prompt

    for page in relevant_pages:
        context_parts.append(f"""
        PAGE TROUVÉE:
        Titre: {page.title}
        URL: /{page.slug}
        Description: {page.meta_description}
        """)
        # Note: On pourrait scanner page.sections.data ici, mais c'est lourd pour un MVP regex.

    # 3. Recherche dans le Blog (Articles)
    relevant_articles = Article.objects.filter(
        status='published'
    ).filter(
        Q(title__icontains=query) | 
        Q(excerpt__icontains=query)
    )[:3]

    for article in relevant_articles:
        context_parts.append(f"""
        ARTICLE DE BLOG:
        Titre: {article.title}
        URL: /blog/{article.slug}
        Résumé: {article.excerpt}
        """)

    return "\n---\n".join(context_parts)

def get_ai_response(user_query: str, conversation_uuid=None) -> dict:
    """
    Orchestre la réponse IA.
    Retourne un dict {'reply': str, 'conversation_uuid': str}
    """
    # 1. Gestion de la session (Conversation)
    if conversation_uuid:
        try:
            conversation = Conversation.objects.get(uuid=conversation_uuid)
        except Conversation.DoesNotExist:
            conversation = Conversation.objects.create()
    else:
        conversation = Conversation.objects.create()

    # 2. Sauvegarde du message User
    ChatMessage.objects.create(
        conversation=conversation,
        role='user',
        content=user_query
    )
    conversation.message_count += 1
    conversation.save()

    # 3. Construction du Prompt (RAG)
    context_data = find_context_in_db(user_query)
    
    # Récupération du nom du site pour le prompt système
    site_config = GlobalSettings.objects.first()
    site_name = site_config.site_name if site_config else "Site Web"
    
    system_prompt = prompts.get_base_prompt(site_name=site_name)
    
    if context_data:
        full_user_message = f"""
        CONTEXTE TROUVÉ DANS LA BASE DE DONNÉES :
        {context_data}

        QUESTION UTILISATEUR :
        {user_query}
        """
    else:
        full_user_message = user_query # Le prompt système gérera l'absence de contexte

    # 4. Appel API OpenRouter
    messages = [
        {"role": "system", "content": system_prompt},
        # On pourrait réinjecter l'historique récent ici :
        # *conversation.messages.order_by('-created_at')[:3]...
        {"role": "user", "content": full_user_message}
    ]

    try:
        completion = client.chat.completions.create(
            model=settings.OPENROUTER_MODEL,
            messages=messages,
            temperature=0.7,
            max_tokens=300,
        )
        ai_reply = completion.choices[0].message.content.strip()

    except Exception as e:
        logger.error(f"OpenRouter Error: {e}")
        ai_reply = "Je rencontre une petite difficulté technique pour accéder à mon cerveau numérique. Pouvez-vous reformuler ou réessayer dans un instant ?"

    # 5. Sauvegarde de la réponse IA
    ChatMessage.objects.create(
        conversation=conversation,
        role='assistant',
        content=ai_reply,
        used_context=bool(context_data)
    )

    return {
        "reply": ai_reply,
        "conversation_uuid": str(conversation.uuid)
    }